# Functions for getting Virtualmin technical support

use POSIX;

BEGIN { push(@INC, ".."); };
eval "use WebminCore;";
&init_config();

&foreign_require("at", "at-lib.pl");
&foreign_require("cron", "cron-lib.pl");
&foreign_require("acl", "acl-lib.pl");
&foreign_require("mailboxes", "mailboxes-lib.pl");

$disable_login_cmd = "$module_config_directory/disable-login.pl";
$login_webmin_user = "remote-support";
$virtualmin_remote_email = $config{'remote_email'} ||
	"ilia\@virtualmin.dev";
$virtualmin_community = "https://forum.virtualmin.com";

# get_virtualmin_ssh_keys()
# Returns a list of SSH authorized keys for Virtualmin tech support, currently
# active for root.
sub get_virtualmin_ssh_keys
{
my @allkeys = &read_authorized_keys(&get_authorized_keys_file());
my @vkeys = grep { $_->{'cmt'} =~ /^Virtualmin:|^Virtualmin\s+Support/i } @allkeys;
return @vkeys;
}

# save_virtualmin_ssh_keys(&key, ...)
# Updates the Virtualmin tech support keys to the list provided, which may
# be empty.
sub save_virtualmin_ssh_keys
{
my @vkeys = @_;
my @allkeys = &read_authorized_keys(&get_authorized_keys_file());
my @nonvkeys = grep { $_->{'cmt'} !~ /^Virtualmin:|^Virtualmin\s+Support/i } @allkeys;
&write_authorized_keys(&get_authorized_keys_file(), @nonvkeys, @vkeys);
}

# read_authorized_keys(file)
# Returns keys from some file
sub read_authorized_keys
{
my ($file) = @_;
my @rv;
my $lnum = 0;
open(AUTHS, "<".$file);
while(<AUTHS>) {
	s/\r|\n//g;
	my $cmt;
	if (s/#\s*(.*)$//) {
		# Extract comment
		$cmt = $1;
		}
	if (/^((.*)\s+)?(\d+)\s+(\d+)\s+(\d+)\s+(\S+)$/) {
		# SSH1 style line
		my $auth = { 'name' => $6, 'key' => $5, 'exp' => $4,
			     'bits' => $3, 'opts' => $2, 'type' => 1,
			     'line' => $lnum, 'index' => scalar(@rv),
			     'cmt' => $cmt };
		$auth->{'opts'} =~ s/\s+$//;
		push(@rv, $auth);
		}
	elsif (/^((.*)\s+)?([a-z]\S+)\s+(\S+)\s+(\S+)/) {
		# SSH2 style line
		my $auth = { 'name' => $5, 'key' => $4, 'opts' => $2,
			     'keytype' => $3, 'type' => 2,
			     'line' => $lnum, 'index' => scalar(@rv),
			     'cmt' => $cmt };
		$auth->{'opts'} =~ s/\s+$//;
		push(@rv, $auth);
		}
	elsif (/\@/) {
		# Some other line
		my $line = { 'text' => $_, 
			     'line' => $lnum, 'index' => scalar(@rv),
			     'cmt' => $cmt };
		push(@rv, $auth);
		}
	$lnum++;
	}
close(AUTHS);
return @rv;
}

# write_authorized_keys(file, &key, ...)
# Write all provided SSH authorized keys out to a file
sub write_authorized_keys
{
my ($file, @keys) = @_;

# Check if dir exists (such as ~/.ssh)
local $dir = $file;
$dir =~ s/\/[^\/]+$//;
if (!-d $dir) {
	&make_dir($dir, 0700);
	}

# Write out keys
local $existed = -r $file;
&open_tempfile(AUTHS, ">$file");
foreach my $auth (@keys) {
	my $line;
	if ($auth->{'type'} == 2) {
		# SSH 2 format line
		$line = join(" ", $auth->{'opts'} ? ( $auth->{'opts'} ) : ( ),
				  $auth->{'keytype'}, $auth->{'key'},
				  $auth->{'name'} );
		}
	elsif ($auth->{'type'} == 1) {
		# SSH 1 format line
		$line = join(" ", $auth->{'opts'} ? ( $auth->{'opts'} ) : ( ),
				  $auth->{'bits'}, $auth->{'exp'},
				  $auth->{'key'}, $auth->{'name'} );
		}
	elsif ($auth->{'text'}) {
		# Some other line
		$line = $auth->{'text'};
		}
	if ($line) {
		$line .= " # ".$auth->{'cmt'} if ($auth->{'cmt'});
		&print_tempfile(AUTHS, $line,"\n");
		}
	}
&close_tempfile(AUTHS);

# Set sensible permissions
if (!$existed) {
	&set_ownership_permissions(undef, undef, 0600, $file);
	}
}

# get_authorized_keys_file()
# Returns the full path to root's authorized keys file
sub get_authorized_keys_file
{
my @uinfo = getpwnam($config{'user'} || "root");
my $kfile = "$uinfo[7]/.ssh/authorized_keys";
my $kfile2 = "$uinfo[7]/.ssh/authorized_keys2";
return -r $kfile ? $kfile : -r $kfile2 ? $kfile2 : $kfile;
}

# available_virtualmin_ssh_keys()
# Returns a list of keys built into the module for tech support
sub available_virtualmin_ssh_keys
{
return &read_authorized_keys("$module_root_directory/virtualmin_keys");
}

# get_disable_login_job()
# Returns the at job for disabling the remote login, if any
sub get_disable_login_job
{
my @jobs = &at::list_atjobs();
my ($job) = grep { $_->{'realcmd'} =~ /\Q$disable_login_cmd\E/ } @jobs;
return $job;
}

# get_virtualmin_serial()
# Returns the serial number and key for the installed Virtualmi, or undef if
# missing or GPL.
sub get_virtualmin_serial
{
my %serial;
&read_env_file("/etc/virtualmin-license", \%serial);
if ($serial{'SerialNumber'} && $serial{'SerialNumber'} ne 'GPL') {
	return ($serial{'SerialNumber'}, $serial{'LicenseKey'});
	}
return ( );
}

# post_http_connection(&hostname, port, page, &cgi-params, &out, &err)
# Makes an HTTP post to some URL, sending the given CGI parameters as data.
sub post_http_connection
{
local ($host, $port, $page, $params, $out, $err) = @_;

local $oldproxy = $gconfig{'http_proxy'};	# Proxies mess up connection
$gconfig{'http_proxy'} = '';			# to the IP explicitly
local $h = &make_http_connection($host, $port, 1, "POST", $page);
$gconfig{'http_proxy'} = $oldproxy;
if (!ref($h)) {
	$$err = $h;
	return 0;
	}
&write_http_connection($h, "Host: $host\r\n");
&write_http_connection($h, "User-agent: Webmin\r\n");
&write_http_connection($h, "Content-type: application/x-www-form-urlencoded\r\n");
&write_http_connection($h, "Content-length: ".length($params)."\r\n");
&write_http_connection($h, "\r\n");
&write_http_connection($h, "$params\r\n");

# Read back the results
&complete_http_download($h, $out, $err, undef, 0, $host, $port);
}

# get_external_ip_address()
# Returns the IP address of this system, as seen by other hosts on the Internet.
sub get_external_ip_address
{
local $url = "http://software.virtualmin.com/cgi-bin/ip.cgi";
local ($host, $port, $page, $ssl) = &parse_http_url($url);
local ($out, $error);
&http_download($host, $port, $page, \$out, \$error, undef, $ssl,
	       undef, undef, 5, 0, 1);
$out =~ s/\r|\n//g;
return $error ? undef : $out;
}

# get_structured_sysinfo()
# Returns system information in a simple array with title stored
# as a string, and data as stored as a hash ref with key / value
sub get_structured_sysinfo
{
# Get all info
my $is_virtualmin;
my $is_cloudmin;
$is_virtualmin++ if (&foreign_check("virtual-server"));
$is_cloudmin++ if (&foreign_check("server-manager"));
my @sysinfo_complete = &list_combined_system_info({ 'noquotas' => 1, 'nostatus' => 1, 'noupdates' => 1, 'nobw' => 1, 'noips' => 1 });
my @sysinfo = grep { $_->{'id'} eq 'sysinfo' } @sysinfo_complete;
my $info    = @sysinfo[0]->{'raw'}[0];

#
# Resources usage
#
my @sysinfo_resource_usage;
if ($info->{'cpu'} || @{ $info->{'mem'} } || $info->{'disk_total'}) {
    push(@sysinfo_resource_usage, 'Resources usage');
    }
if ($info->{'cpu'}) {
    my @c = @{ $info->{'cpu'} };
    push(@sysinfo_resource_usage, { 'CPU usage' => int($c[0] + $c[1] + $c[3]). "%" })
    }

if ($info->{'mem'}) {
    my (@m, $mem_percent, $virt_percent) = (@{ $info->{'mem'} });
    if (@m && $m[0]) {
        $mem_percent = int(($m[0] - $m[1]) / $m[0] * 100);
        push(@sysinfo_resource_usage,
            { 'Real memory usage' => $mem_percent . "%" . ($m[3] ? " (" . nice_size($m[3] * 1024, -1) . " total)" : '') });
        }
    if (@m && $m[2]) {
        $virt_percent = int(($m[2] - $m[3]) / $m[2] * 100);
        push(@sysinfo_resource_usage,
            { 'Virtual memory usage' => $mem_percent . "%" . ($m[2] ? " (" . nice_size($m[2] * 1024, -1) . " total)" : '') });
        }
    }
if ($info->{'disk_total'}) {
    my ($total, $free) = ($info->{'disk_total'}, $info->{'disk_free'});
    $disk_percent = int(($total - $free) / $total * 100);
    push(@sysinfo_resource_usage,
        { 'Disk Usage' => $disk_percent . "%" . (" (" . nice_size($info->{'disk_total'}, -1) . " total)") });
    }
push(@structured_sysinfo, @sysinfo_resource_usage);

#
# Operating System Info
#
my @sysinfo_sysinfo;
push(@sysinfo_sysinfo, 'Server information');
push(@sysinfo_sysinfo, { 'System hostname' => &get_display_hostname() });

if ($gconfig{'os_version'} eq '*') {
    $os = $gconfig{'real_os_type'};
    }
else {
    $os = $gconfig{'real_os_type'} . ' ' . $gconfig{'real_os_version'};
    }
push(@sysinfo_sysinfo, { 'Operating system' => $os });

# System time
my $loc_time = localtime(time());
my $time_zone = strftime("%z, %Z", localtime());
push(@sysinfo_sysinfo, { 'Time on system' => $loc_time . ($time_zone ? " ($time_zone)" : "") });

# Uptime & running proc count
if (foreign_check("proc") && foreign_available("proc")) {
    foreign_require("proc");
    my @system_uptime = defined(&proc::get_system_uptime) ? proc::get_system_uptime() : ();
    if (@system_uptime) {
        my ($day, $hour, $minute) = @system_uptime;
        my $uptime_text;
        if ($day) {
            $uptime_text = "$day days, $hour hours, $minute minutes";
            }
        elsif ($minute && $hour) {
            $uptime_text = "$hour hours, $minute minutes";
            }
        elsif ($minute) {
            $uptime_text = "$minute minutes";
            }
        if ($current_lang eq 'en') {
            if ($day == 1) {
                $uptime_text =~ s/ys/y/;
                }
            if ($hour == 1) {
                $uptime_text =~ s/rs/r/;
                }
            if ($minute == 1) {
                $uptime_text =~ s/es/e/;
                }
            }
        push(@sysinfo_sysinfo, { 'System uptime' => "$uptime_text" });
        }
    }


# Kernel and arch
if ($info->{'kernel'}) {
    push(@sysinfo_sysinfo,
        { 'Kernel and arch' => "$info->{'kernel'}->{'os'} $info->{'kernel'}->{'version'} on $info->{'kernel'}->{'arch'}" });
    }

# CPU info
if ($info->{'load'}) {
    my @c = @{ $info->{'load'} };
    if (@c > 3) {
        push(@sysinfo_sysinfo, { 'CPU information' => "$c[4], $c[7] cores" });
        }
    }

# Load average
if ($info->{'load'}) {
    my @c = @{ $info->{'load'} };
    if (@c) {
        push(@sysinfo_sysinfo, { 'CPU load averages' => "$c[0] (1 min) $c[1] (5 mins) $c[2] (15 mins)" });
        }
    }

# Uptime & running proc count
if (foreign_check("proc") && foreign_available("proc")) {
    foreign_require("proc");
    # Running processes count
    my @procs = proc::list_processes();
    $running_proc = scalar(@procs);
    push(@sysinfo_sysinfo, { 'Running processes' => "$running_proc" });
    }

# Package updates
if ($info->{'poss'}) {
    my @poss = @{ $info->{'poss'} };
    my @secs = grep {$_->{'security'}} @poss;
    my $poss = scalar(@poss);
    my $secs = scalar(@secs);
    my $msg;
    if ($poss && $secs) {
        $msg = "$poss available ($sec security updates)";
        }
    elsif ($poss) {
        $msg = "$poss available";
        }
    push(@sysinfo_sysinfo, { 'Package updates' => "$msg" }) if ($msg);
    }
push(@structured_sysinfo, @sysinfo_sysinfo);

#
# Servers Status (very slow due to `list_php_fpm_configs()` in `startstop_web()`)
#
if ($is_virtualmin) {
	my @start_stop_links = &virtual_server::get_startstop_links();
	my @sysinfo_servers_status;
	push(@sysinfo_servers_status, 'Servers Status');
	push(@sysinfo_servers_status, map { {$_->{'name'} => ($_->{'status'} ? "Running" : "Stopped" ) } } @start_stop_links);
	push(@structured_sysinfo, @sysinfo_servers_status);
}

#
# Virtualmin counts
#
my @ftypes = grep {$_->{'id'} eq 'ftypes'} @sysinfo_complete;
if (scalar(@ftypes) && $ftypes[0]->{'table'}) {
    my @sysinfo_vm_counts;
    push(@sysinfo_vm_counts, 'Virtualmin counts');
    my $sysinfo_vm_counts_arr = $ftypes[0]->{'table'};
    push(@sysinfo_vm_counts, map { {$_->{'desc'} => $_->{'value'} } } @$sysinfo_vm_counts_arr);
    push(@structured_sysinfo, @sysinfo_vm_counts);
    }

#
# Software versions
#
if ($is_virtualmin) {
	$main::sysinfo_virtualmin_self = 1;
	my @software_versions;
	push(@software_versions, 'Software versions');
	my @software_versions_all = &virtual_server::sysinfo_virtualmin();
	foreach my $f (@virtual_server::features) {
	    if ($virtual_server::config{$f}) {
	        my $ifunc = "virtual_server::sysinfo_$f";
	        if (defined(&$ifunc)) {
	            push(@software_versions_all, &$ifunc());
	            }
	        }
	    }

	push(@software_versions, map { {$_->[0] => $_->[1] } } @software_versions_all);
	push(@structured_sysinfo, @software_versions);
	}
return \@structured_sysinfo;
}
# get_markdown_url_param(&structured_sysinfo)
# Returns sysinfo in markdown ready to
# be added as part of the URL param 
sub get_markdown_url_param
{
my ($sysinfo) = @_;
my ($markdown, $current_section);
$markdown .= "\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n<!---\n";
$markdown .= "BELOW TEXT CONTAINS YOUR SYSTEM INFORMATION AND SHOULD BE LEFT INTACT\n";
$markdown .= "-->\n";
$markdown .= "\n<br>\n\n---\n<br>\n<details>\n";
my $hostname = &get_display_hostname();
my $summary_text = $hostname ? "$hostname system information" : 'System information';
$markdown .= "<summary><strong>$summary_text</strong></summary>\n\n";
for (my $i = 0; $i <= $#$sysinfo; $i++) {
    my $line = $sysinfo->[$i]; 

    # Title
    if (ref($line) ne 'HASH') {
        $markdown .= ($i != 0 ? "\n|" : '|') . uc($line) . "| |\n|---|---|\n";
        }
    # Data
    else {
        my @label = keys(%{$line});
        my $value = $line->{$label[0]};

        $markdown .= "|" . &trim($label[0]) . " |" . &trim($value) . "|\n";
        }
    }
$markdown .= "\n</details>\n";
$markdown .= "<br>";
$markdown .= "\n<!---\n";
$markdown .= "SCROLL TO THE VERY TOP OF THE PAGE TO FILL YOUR TICKET\n";
$markdown .= "-->\n";
return &urlize($markdown);
}
1;

